<?php
namespace WPTX_AIBD;

if (!defined('ABSPATH')) { exit; }

final class WPTX_AIBD_Updater
{
    private const UPDATE_JSON_URL = 'https://wptechs.net/plugins/ai-share-buttons/update.json';
    private const CACHE_KEY = 'wptx_aibd_update_json_v1';
    private const CACHE_TTL = 12 * HOUR_IN_SECONDS;

    private string $plugin_basename;
    private string $plugin_slug;
    private string $current_version;

    public function __construct(string $plugin_file, string $current_version)
    {
        $this->plugin_basename = plugin_basename($plugin_file);
        $this->plugin_slug = dirname($this->plugin_basename);
        $this->current_version = $current_version;

        if (is_admin()) {
            add_filter('pre_set_site_transient_update_plugins', [$this, 'inject_update']);
            add_filter('plugins_api', [$this, 'plugins_api'], 10, 3);
        }
    }

    public function inject_update($transient)
    {
        if (!is_object($transient) || empty($transient->checked) || !is_array($transient->checked)) {
            return $transient;
        }

        if (!isset($transient->checked[$this->plugin_basename])) {
            return $transient;
        }

        $info = $this->fetch_update_info();
        if (!$info) {
            return $transient;
        }

        $remote_version = isset($info['version']) ? (string)$info['version'] : '';
        $package = isset($info['download_url']) ? (string)$info['download_url'] : '';

        if ($remote_version === '' || $package === '') {
            return $transient;
        }

        if (version_compare($remote_version, $this->current_version, '<=')) {
            return $transient;
        }

        $item = (object)[
            'slug' => $this->plugin_slug,
            'plugin' => $this->plugin_basename,
            'new_version' => $remote_version,
            'package' => $package,
            'url' => isset($info['homepage']) ? (string)$info['homepage'] : '',
            'tested' => isset($info['tested']) ? (string)$info['tested'] : '',
            'requires' => isset($info['requires']) ? (string)$info['requires'] : '',
        ];

        $transient->response[$this->plugin_basename] = $item;
        return $transient;
    }

    public function plugins_api($result, $action, $args)
    {
        if ($action !== 'plugin_information') {
            return $result;
        }
        if (!is_object($args) || empty($args->slug)) {
            return $result;
        }
        if ($args->slug !== $this->plugin_slug && $args->slug !== $this->plugin_basename) {
            return $result;
        }

        $info = $this->fetch_update_info();
        if (!$info) {
            return $result;
        }

        $sections = isset($info['sections']) && is_array($info['sections']) ? $info['sections'] : [];
        $desc = isset($sections['description']) ? (string)$sections['description'] : '';

        $out = (object)[
            'name' => 'AI Share Buttons',
            'slug' => $this->plugin_slug,
            'version' => isset($info['version']) ? (string)$info['version'] : $this->current_version,
            'author' => 'WPTechs.NET',
            'homepage' => isset($info['homepage']) ? (string)$info['homepage'] : '',
            'requires' => isset($info['requires']) ? (string)$info['requires'] : '',
            'tested' => isset($info['tested']) ? (string)$info['tested'] : '',
            'last_updated' => isset($info['last_updated']) ? (string)$info['last_updated'] : '',
            'download_link' => isset($info['download_url']) ? (string)$info['download_url'] : '',
            'sections' => [
                'description' => $desc !== '' ? $desc : 'Adds Summarize & Save to AI buttons to your content.',
            ],
        ];

        return $out;
    }

    public static function clear_cache(): void
    {
        delete_transient(self::CACHE_KEY);
        delete_site_transient('update_plugins');
    }

    private function fetch_update_info(): ?array
    {
        $cached = get_transient(self::CACHE_KEY);
        if (is_array($cached)) {
            return $cached;
        }

        $url = self::UPDATE_JSON_URL;

        $parts = wp_parse_url($url);
        if (empty($parts['scheme']) || strtolower((string)$parts['scheme']) !== 'https') {
            return null;
        }
        if (empty($parts['host']) || strtolower((string)$parts['host']) !== 'wptechs.net') {
            return null;
        }

        $resp = wp_remote_get($url, [
            'timeout' => 8,
            'redirection' => 0,
            'headers' => [
                'Accept' => 'application/json',
                'User-Agent' => 'AI-Share-Buttons/' . $this->current_version . '; ' . home_url('/'),
            ],
        ]);

        if (is_wp_error($resp)) {
            return null;
        }

        $code = (int) wp_remote_retrieve_response_code($resp);
        if ($code != 200) {
            return null;
        }

        $body = (string) wp_remote_retrieve_body($resp);
        $data = json_decode($body, true);
        if (!is_array($data)) {
            return null;
        }

        $dl = isset($data['download_url']) ? (string)$data['download_url'] : '';
        if ($dl === '') { return null; }
        $dlp = wp_parse_url($dl);
        if (empty($dlp['scheme']) || strtolower((string)$dlp['scheme']) !== 'https') { return null; }
        if (empty($dlp['host']) || strtolower((string)$dlp['host']) !== 'wptechs.net') { return null; }
        if (substr($dl, -4) !== '.zip') { return null; }

        set_transient(self::CACHE_KEY, $data, self::CACHE_TTL);
        return $data;
    }
}
