<?php
/*
Plugin Name: Blogger Checklist
Description: A robust tool for managing blog post checklists, featuring fields for social media assets, newsletter inclusion, video tracking, and notes. Built by the WPTechs LLC team to streamline content workflows for WordPress sites.
Version: 2.4
Author: WPTechs LLC
Author URI: https://wptechs.net
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: wptechs-blogger-checklist
*/

if (!defined('ABSPATH')) {
    exit;
}

final class WPTX_Blogger_Checklist
{
    public const VERSION = '2.4';
    public const TEXT_DOMAIN = 'wptechs-blogger-checklist';
    private const MENU_SLUG_MAIN  = 'blogger-checklist';
    private const MENU_SLUG_STATUS = 'blogger-checklist-status';
    private const MENU_SLUG_TOOLS  = 'blogger-checklist-tools';
    private const OPT_DASHBOARD_CLEANER = 'blogger_checklist_dashboard_cleaner';
    private const OPT_ADMINBAR_CLEANER  = 'blogger_checklist_admin_bar_cleaner';
    private const UPDATE_VERSION_URL = 'https://wptechs.net/plugins/blogger-checklist/version.txt';
    private const UPDATE_ZIP_PATTERN = 'https://wptechs.net/plugins/blogger-checklist/blogger-checklist-%s.zip';
    private static ?self $instance = null;

    public static function instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        add_action('init', [$this, 'load_textdomain']);
        register_activation_hook(__FILE__, [$this, 'activate']);
        add_action('transition_post_status', [$this, 'on_transition_post_status'], 10, 3);
        add_action('admin_menu', [$this, 'admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_action('admin_post_wptx_bc_export', [$this, 'handle_export_csv']);
        add_action('admin_post_wptx_bc_save_row', [$this, 'handle_save_row']);
        add_action('admin_post_wptx_bc_check_updates', [$this, 'handle_check_updates']);
        add_action('admin_post_wptx_bc_update_now', [$this, 'handle_update_now']);
        add_action('admin_post_wptx_bc_save_tools', [$this, 'handle_save_tools']);
        add_action('admin_post_wptx_bc_delete_pending_comments', [$this, 'handle_delete_pending_comments']);
        add_action('admin_post_wptx_bc_delete_unused_tags', [$this, 'handle_delete_unused_tags']);
        add_action('admin_post_wptx_bc_delete_unused_categories', [$this, 'handle_delete_unused_categories']);
        add_action('wp_dashboard_setup', [$this, 'maybe_clean_dashboard'], 999);
        add_action('admin_bar_menu', [$this, 'maybe_clean_admin_bar'], 999);
        add_action('admin_post_wptx_bc_empty_trash', [$this, 'handle_empty_trash']);
        add_action('admin_post_wptx_bc_delete_spam_comments', [$this, 'handle_delete_spam_comments']);
        add_action('admin_post_wptx_bc_clear_expired_transients', [$this, 'handle_clear_expired_transients']);
    }

    public function load_textdomain(): void
    {
        load_plugin_textdomain(self::TEXT_DOMAIN, false, dirname(plugin_basename(__FILE__)) . '/languages');
    }

    private function table_name(): string
    {
        global $wpdb;
        return $wpdb->prefix . 'blogger_checklist';
    }

    public function activate(): void
    {
        $this->create_or_update_table();
        $this->populate_existing_published_posts_fast();
    }

    private function create_or_update_table(): void
    {
        global $wpdb;

        $table = $this->table_name();
        $charset_collate = $wpdb->get_charset_collate();

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $sql = "CREATE TABLE {$table} (
            id BIGINT(20) unsigned NOT NULL AUTO_INCREMENT,
            post_id BIGINT(20) unsigned NOT NULL,
            pinterest_image_1 VARCHAR(255) NOT NULL DEFAULT '',
            pinterest_image_2 VARCHAR(255) NOT NULL DEFAULT '',
            pinterest_image_3 VARCHAR(255) NOT NULL DEFAULT '',
            instagram_image VARCHAR(255) NOT NULL DEFAULT '',
            newsletter_featured TINYINT(1) NOT NULL DEFAULT 0,
            video TINYINT(1) NOT NULL DEFAULT 0,
            notes LONGTEXT NOT NULL,
            PRIMARY KEY  (id),
            UNIQUE KEY post_id (post_id)
        ) {$charset_collate};";

        dbDelta($sql);

        $wpdb->query("UPDATE {$table} SET notes = '' WHERE notes IS NULL");
    }

    private function populate_existing_published_posts_fast(): void
    {
        global $wpdb;

        $table = $this->table_name();

        $wpdb->query("
            INSERT IGNORE INTO {$table} (post_id, notes)
            SELECT ID, ''
            FROM {$wpdb->posts}
            WHERE post_type = 'post' AND post_status = 'publish'
        ");
    }

    public function on_transition_post_status(string $new_status, string $old_status, \WP_Post $post): void
    {
        if ($post->post_type !== 'post') {
            return;
        }
        if ($new_status !== 'publish' || $old_status === 'publish') {
            return;
        }

        $this->ensure_row_for_post((int) $post->ID);
    }

    private function ensure_row_for_post(int $post_id): void
    {
        global $wpdb;
        $table = $this->table_name();

        $wpdb->query(
            $wpdb->prepare(
                "INSERT IGNORE INTO {$table} (post_id, notes) VALUES (%d, %s)",
                $post_id,
                ''
            )
        );
    }

    public function admin_menu(): void
    {
        add_menu_page(
            __('Blogger Checklist', self::TEXT_DOMAIN),
            __('Blogger Checklist', self::TEXT_DOMAIN),
            'manage_options',
            self::MENU_SLUG_MAIN,
            [$this, 'render_main_page'],
            'dashicons-list-view',
            20
        );

        add_submenu_page(
            self::MENU_SLUG_MAIN,
            __('Status', self::TEXT_DOMAIN),
            __('Status', self::TEXT_DOMAIN),
            'manage_options',
            self::MENU_SLUG_STATUS,
            [$this, 'render_status_page']
        );

        add_submenu_page(
            self::MENU_SLUG_MAIN,
            __('Neat Tools', self::TEXT_DOMAIN),
            __('Neat Tools', self::TEXT_DOMAIN),
            'manage_options',
            self::MENU_SLUG_TOOLS,
            [$this, 'render_tools_page']
        );
    }

    public function enqueue_admin_assets(string $hook): void
    {
        $allowed = [
            'toplevel_page_' . self::MENU_SLUG_MAIN,
            self::MENU_SLUG_MAIN . '_page_' . self::MENU_SLUG_STATUS,
            self::MENU_SLUG_MAIN . '_page_' . self::MENU_SLUG_TOOLS,
        ];

        if (!in_array($hook, $allowed, true)) {
            return;
        }

        wp_enqueue_style(
            'wptechs-blogger-checklist',
            plugins_url('style.css', __FILE__),
            [],
            self::VERSION
        );
    }

    public function render_main_page(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', self::TEXT_DOMAIN));
        }

        global $wpdb;
        $table = $this->table_name();

        $per_page = 25;
        $paged    = isset($_GET['paged']) ? max(1, (int) $_GET['paged']) : 1;
        $offset   = ($paged - 1) * $per_page;

        $search            = isset($_GET['s']) ? sanitize_text_field(wp_unslash($_GET['s'])) : '';
        $status_filter     = isset($_GET['status_filter']) ? sanitize_text_field(wp_unslash($_GET['status_filter'])) : 'all';
        $pinterest_filter  = isset($_GET['pinterest_filter']) ? sanitize_text_field(wp_unslash($_GET['pinterest_filter'])) : 'all';
        $instagram_filter  = isset($_GET['instagram_filter']) ? sanitize_text_field(wp_unslash($_GET['instagram_filter'])) : 'all';
        $newsletter_filter = isset($_GET['newsletter_filter']) ? sanitize_text_field(wp_unslash($_GET['newsletter_filter'])) : 'all';
        $video_filter      = isset($_GET['video_filter']) ? sanitize_text_field(wp_unslash($_GET['video_filter'])) : 'all';

        $where_sql = "WHERE p.post_type = 'post' AND p.post_status = 'publish'";
        $args = [];

        if ($search !== '') {
            $where_sql .= " AND p.post_title LIKE %s";
            $args[] = '%' . $wpdb->esc_like($search) . '%';
        }

        $complete_sql = "(bc.pinterest_image_1 <> '' AND bc.pinterest_image_2 <> '' AND bc.pinterest_image_3 <> '' AND bc.instagram_image <> '' AND bc.newsletter_featured = 1 AND bc.video = 1)";
        if ($status_filter === 'complete') {
            $where_sql .= " AND {$complete_sql}";
        } elseif ($status_filter === 'incomplete') {
            $where_sql .= " AND NOT {$complete_sql}";
        }

        if ($pinterest_filter === 'missing') {
            $where_sql .= " AND (bc.pinterest_image_1 = '' OR bc.pinterest_image_2 = '' OR bc.pinterest_image_3 = '')";
        } elseif ($pinterest_filter === 'present') {
            $where_sql .= " AND (bc.pinterest_image_1 <> '' AND bc.pinterest_image_2 <> '' AND bc.pinterest_image_3 <> '')";
        }

        if ($instagram_filter === 'missing') {
            $where_sql .= " AND bc.instagram_image = ''";
        } elseif ($instagram_filter === 'present') {
            $where_sql .= " AND bc.instagram_image <> ''";
        }

        if ($newsletter_filter === 'missing') {
            $where_sql .= " AND bc.newsletter_featured = 0";
        } elseif ($newsletter_filter === 'present') {
            $where_sql .= " AND bc.newsletter_featured = 1";
        }

        if ($video_filter === 'missing') {
            $where_sql .= " AND bc.video = 0";
        } elseif ($video_filter === 'present') {
            $where_sql .= " AND bc.video = 1";
        }

        $this->populate_existing_published_posts_fast();

        $base_sql = "
            FROM {$table} bc
            INNER JOIN {$wpdb->posts} p ON bc.post_id = p.ID
            {$where_sql}
        ";

        $count_sql = "SELECT COUNT(*) {$base_sql}";
        $total_items = ($args)
            ? (int) $wpdb->get_var($wpdb->prepare($count_sql, $args))
            : (int) $wpdb->get_var($count_sql);

        $list_sql = "
            SELECT bc.*, p.post_title, p.ID as post_id
            {$base_sql}
            ORDER BY p.post_date DESC
            LIMIT %d, %d
        ";

        $list_args = $args;
        $list_args[] = $offset;
        $list_args[] = $per_page;

        $items = $wpdb->get_results($wpdb->prepare($list_sql, $list_args));

        $current_args = array_intersect_key(
            $_GET,
            array_flip(['status_filter', 'pinterest_filter', 'instagram_filter', 'newsletter_filter', 'video_filter', 's'])
        );
        $current_args = array_map('sanitize_text_field', wp_unslash($current_args));

        $export_url = wp_nonce_url(
            admin_url('admin-post.php?action=wptx_bc_export'),
            'wptx_bc_export'
        );

        if (!empty($current_args)) {
            $export_url = add_query_arg($current_args, $export_url);
        }

        ?>
        <div class="wrap blogger-checklist-wrap">
            <h1><?php echo esc_html__('Blogger Checklist', self::TEXT_DOMAIN); ?></h1>

            <form method="get" class="checklist-filters">
                <input type="hidden" name="page" value="<?php echo esc_attr(self::MENU_SLUG_MAIN); ?>">

                <label for="status_filter"><?php echo esc_html__('Status:', self::TEXT_DOMAIN); ?></label>
                <select name="status_filter" id="status_filter" onchange="this.form.submit()">
                    <option value="all" <?php selected($status_filter, 'all'); ?>><?php echo esc_html__('All', self::TEXT_DOMAIN); ?></option>
                    <option value="complete" <?php selected($status_filter, 'complete'); ?>><?php echo esc_html__('Complete', self::TEXT_DOMAIN); ?></option>
                    <option value="incomplete" <?php selected($status_filter, 'incomplete'); ?>><?php echo esc_html__('Incomplete', self::TEXT_DOMAIN); ?></option>
                </select>

                <label for="pinterest_filter"><?php echo esc_html__('Pinterest:', self::TEXT_DOMAIN); ?></label>
                <select name="pinterest_filter" id="pinterest_filter" onchange="this.form.submit()">
                    <option value="all" <?php selected($pinterest_filter, 'all'); ?>><?php echo esc_html__('All', self::TEXT_DOMAIN); ?></option>
                    <option value="missing" <?php selected($pinterest_filter, 'missing'); ?>><?php echo esc_html__('Missing', self::TEXT_DOMAIN); ?></option>
                    <option value="present" <?php selected($pinterest_filter, 'present'); ?>><?php echo esc_html__('Present', self::TEXT_DOMAIN); ?></option>
                </select>

                <label for="instagram_filter"><?php echo esc_html__('Instagram:', self::TEXT_DOMAIN); ?></label>
                <select name="instagram_filter" id="instagram_filter" onchange="this.form.submit()">
                    <option value="all" <?php selected($instagram_filter, 'all'); ?>><?php echo esc_html__('All', self::TEXT_DOMAIN); ?></option>
                    <option value="missing" <?php selected($instagram_filter, 'missing'); ?>><?php echo esc_html__('Missing', self::TEXT_DOMAIN); ?></option>
                    <option value="present" <?php selected($instagram_filter, 'present'); ?>><?php echo esc_html__('Present', self::TEXT_DOMAIN); ?></option>
                </select>

                <label for="newsletter_filter"><?php echo esc_html__('Newsletter:', self::TEXT_DOMAIN); ?></label>
                <select name="newsletter_filter" id="newsletter_filter" onchange="this.form.submit()">
                    <option value="all" <?php selected($newsletter_filter, 'all'); ?>><?php echo esc_html__('All', self::TEXT_DOMAIN); ?></option>
                    <option value="missing" <?php selected($newsletter_filter, 'missing'); ?>><?php echo esc_html__('Missing', self::TEXT_DOMAIN); ?></option>
                    <option value="present" <?php selected($newsletter_filter, 'present'); ?>><?php echo esc_html__('Present', self::TEXT_DOMAIN); ?></option>
                </select>

                <label for="video_filter"><?php echo esc_html__('Video:', self::TEXT_DOMAIN); ?></label>
                <select name="video_filter" id="video_filter" onchange="this.form.submit()">
                    <option value="all" <?php selected($video_filter, 'all'); ?>><?php echo esc_html__('All', self::TEXT_DOMAIN); ?></option>
                    <option value="missing" <?php selected($video_filter, 'missing'); ?>><?php echo esc_html__('Missing', self::TEXT_DOMAIN); ?></option>
                    <option value="present" <?php selected($video_filter, 'present'); ?>><?php echo esc_html__('Present', self::TEXT_DOMAIN); ?></option>
                </select>

                <input type="search" name="s" value="<?php echo esc_attr($search); ?>" placeholder="<?php echo esc_attr__('Search posts…', self::TEXT_DOMAIN); ?>">
                <input type="submit" value="<?php echo esc_attr__('Apply', self::TEXT_DOMAIN); ?>" class="button">
            </form>

            <?php if (empty($items)) : ?>
                <p class="bc-empty">
                    <?php echo esc_html__('No posts match your search or filter criteria.', self::TEXT_DOMAIN); ?>
                </p>
            <?php else : ?>
                <div class="checklist-entries">
                    <?php foreach ($items as $item) :
                        $is_complete = (
                            $item->pinterest_image_1 !== '' &&
                            $item->pinterest_image_2 !== '' &&
                            $item->pinterest_image_3 !== '' &&
                            $item->instagram_image !== '' &&
                            (int)$item->newsletter_featured === 1 &&
                            (int)$item->video === 1
                        );

                        $save_url = wp_nonce_url(
                            admin_url('admin-post.php?action=wptx_bc_save_row'),
                            'wptx_bc_save_row'
                        );
                        ?>
                        <div class="checklist-entry">
                            <form method="post" action="<?php echo esc_url($save_url); ?>">
                                <input type="hidden" name="post_id" value="<?php echo (int) $item->post_id; ?>">

                                <?php
                                foreach ($current_args as $k => $v) {
                                    echo '<input type="hidden" name="redirect_' . esc_attr($k) . '" value="' . esc_attr($v) . '">';
                                }
                                echo '<input type="hidden" name="redirect_paged" value="' . (int) $paged . '">';
                                ?>

                                <div class="checklist-field">
                                    <label><?php echo esc_html__('Post Title', self::TEXT_DOMAIN); ?></label>
                                    <span>
                                        <?php echo esc_html($item->post_title ?: ('Post #' . (int)$item->post_id . ' (Deleted)')); ?>
                                        <span class="bc-post-links">
                                            <a class="button button-small" href="<?php echo esc_url(get_edit_post_link((int)$item->post_id, '')); ?>">Edit</a>
                                            <a class="button button-small" target="_blank" rel="noopener noreferrer" href="<?php echo esc_url(get_permalink((int)$item->post_id)); ?>">View</a>
                                        </span>
                                    </span>
                                </div>

                                <div class="checklist-field">
                                    <label for="p1_<?php echo (int)$item->post_id; ?>"><?php echo esc_html__('Pinterest Image 1', self::TEXT_DOMAIN); ?></label>
                                    <input type="text" id="p1_<?php echo (int)$item->post_id; ?>" name="pinterest_image_1" value="<?php echo esc_attr($item->pinterest_image_1); ?>">
                                </div>

                                <div class="checklist-field">
                                    <label for="p2_<?php echo (int)$item->post_id; ?>"><?php echo esc_html__('Pinterest Image 2', self::TEXT_DOMAIN); ?></label>
                                    <input type="text" id="p2_<?php echo (int)$item->post_id; ?>" name="pinterest_image_2" value="<?php echo esc_attr($item->pinterest_image_2); ?>">
                                </div>

                                <div class="checklist-field">
                                    <label for="p3_<?php echo (int)$item->post_id; ?>"><?php echo esc_html__('Pinterest Image 3', self::TEXT_DOMAIN); ?></label>
                                    <input type="text" id="p3_<?php echo (int)$item->post_id; ?>" name="pinterest_image_3" value="<?php echo esc_attr($item->pinterest_image_3); ?>">
                                </div>

                                <div class="checklist-field">
                                    <label for="ig_<?php echo (int)$item->post_id; ?>"><?php echo esc_html__('Instagram Image', self::TEXT_DOMAIN); ?></label>
                                    <input type="text" id="ig_<?php echo (int)$item->post_id; ?>" name="instagram_image" value="<?php echo esc_attr($item->instagram_image); ?>">
                                </div>

                                <div class="checklist-field checkbox">
                                    <label for="nl_<?php echo (int)$item->post_id; ?>"><?php echo esc_html__('Newsletter Featured', self::TEXT_DOMAIN); ?></label>
                                    <input type="checkbox" id="nl_<?php echo (int)$item->post_id; ?>" name="newsletter_featured" <?php checked((int)$item->newsletter_featured, 1); ?>>
                                </div>

                                <div class="checklist-field checkbox">
                                    <label for="vid_<?php echo (int)$item->post_id; ?>"><?php echo esc_html__('Video', self::TEXT_DOMAIN); ?></label>
                                    <input type="checkbox" id="vid_<?php echo (int)$item->post_id; ?>" name="video" <?php checked((int)$item->video, 1); ?>>
                                </div>

                                <div class="checklist-field">
                                    <label for="notes_<?php echo (int)$item->post_id; ?>"><?php echo esc_html__('Notes', self::TEXT_DOMAIN); ?></label>
                                    <textarea id="notes_<?php echo (int)$item->post_id; ?>" name="notes" rows="2"><?php echo esc_textarea($item->notes ?? ''); ?></textarea>
                                </div>

                                <div class="checklist-field">
                                    <label><?php echo esc_html__('Status', self::TEXT_DOMAIN); ?></label>
                                    <span class="status <?php echo $is_complete ? 'complete' : 'incomplete'; ?>">
                                        <?php echo esc_html($is_complete ? __('Complete', self::TEXT_DOMAIN) : __('Incomplete', self::TEXT_DOMAIN)); ?>
                                    </span>
                                </div>

                                <div class="checklist-field actions">
                                    <label><?php echo esc_html__('Actions', self::TEXT_DOMAIN); ?></label>
                                    <button type="submit" class="button button-primary"><?php echo esc_html__('Save', self::TEXT_DOMAIN); ?></button>
                                </div>
                            </form>
                        </div>
                    <?php endforeach; ?>
                </div>

                <?php
                $total_pages = (int) ceil($total_items / $per_page);
                $big = 999999;

                echo '<div class="pagination-links">';
                echo paginate_links([
                    'base'    => str_replace($big, '%#%', esc_url(add_query_arg('paged', $big))),
                    'format'  => '',
                    'current' => $paged,
                    'total'   => max(1, $total_pages),
                    'add_args'=> $current_args,
                ]);
                echo '</div>';
                ?>

                <p class="bc-export">
                    <a class="button" href="<?php echo esc_url($export_url); ?>">
                        <?php echo esc_html__('Export to CSV', self::TEXT_DOMAIN); ?>
                    </a>
                </p>
            <?php endif; ?>
        </div>
        <?php
    }

    public function handle_save_row(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to do that.', self::TEXT_DOMAIN));
        }
        check_admin_referer('wptx_bc_save_row');

        global $wpdb;
        $table = $this->table_name();

        $post_id = isset($_POST['post_id']) ? (int) $_POST['post_id'] : 0;
        if ($post_id <= 0) {
            wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_MAIN));
            exit;
        }

        $this->ensure_row_for_post($post_id);

        $p1 = isset($_POST['pinterest_image_1']) ? esc_url_raw(wp_unslash($_POST['pinterest_image_1'])) : '';
        $p2 = isset($_POST['pinterest_image_2']) ? esc_url_raw(wp_unslash($_POST['pinterest_image_2'])) : '';
        $p3 = isset($_POST['pinterest_image_3']) ? esc_url_raw(wp_unslash($_POST['pinterest_image_3'])) : '';
        $ig = isset($_POST['instagram_image']) ? esc_url_raw(wp_unslash($_POST['instagram_image'])) : '';

        $newsletter = isset($_POST['newsletter_featured']) ? 1 : 0;
        $video      = isset($_POST['video']) ? 1 : 0;

        $notes = isset($_POST['notes']) ? sanitize_textarea_field(wp_unslash($_POST['notes'])) : '';

        $wpdb->update(
            $table,
            [
                'pinterest_image_1'   => $p1,
                'pinterest_image_2'   => $p2,
                'pinterest_image_3'   => $p3,
                'instagram_image'     => $ig,
                'newsletter_featured' => $newsletter,
                'video'               => $video,
                'notes'               => $notes,
            ],
            ['post_id' => $post_id],
            ['%s','%s','%s','%s','%d','%d','%s'],
            ['%d']
        );

        $redirect_args = ['page' => self::MENU_SLUG_MAIN];

        foreach (['status_filter','pinterest_filter','instagram_filter','newsletter_filter','video_filter','s'] as $k) {
            $rk = 'redirect_' . $k;
            if (isset($_POST[$rk]) && $_POST[$rk] !== '') {
                $redirect_args[$k] = sanitize_text_field(wp_unslash($_POST[$rk]));
            }
        }
        if (isset($_POST['redirect_paged'])) {
            $redirect_args['paged'] = max(1, (int) $_POST['redirect_paged']);
        }

        $redirect_url = add_query_arg($redirect_args, admin_url('admin.php'));
        $redirect_url = add_query_arg('bc_saved', '1', $redirect_url);

        wp_safe_redirect($redirect_url);
        exit;
    }

    public function handle_export_csv(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to export.', self::TEXT_DOMAIN));
        }
        check_admin_referer('wptx_bc_export');

        global $wpdb;
        $table = $this->table_name();

        $search            = isset($_GET['s']) ? sanitize_text_field(wp_unslash($_GET['s'])) : '';
        $status_filter     = isset($_GET['status_filter']) ? sanitize_text_field(wp_unslash($_GET['status_filter'])) : 'all';
        $pinterest_filter  = isset($_GET['pinterest_filter']) ? sanitize_text_field(wp_unslash($_GET['pinterest_filter'])) : 'all';
        $instagram_filter  = isset($_GET['instagram_filter']) ? sanitize_text_field(wp_unslash($_GET['instagram_filter'])) : 'all';
        $newsletter_filter = isset($_GET['newsletter_filter']) ? sanitize_text_field(wp_unslash($_GET['newsletter_filter'])) : 'all';
        $video_filter      = isset($_GET['video_filter']) ? sanitize_text_field(wp_unslash($_GET['video_filter'])) : 'all';

        $where_sql = "WHERE p.post_type = 'post' AND p.post_status = 'publish'";
        $args = [];

        if ($search !== '') {
            $where_sql .= " AND p.post_title LIKE %s";
            $args[] = '%' . $wpdb->esc_like($search) . '%';
        }

        $complete_sql = "(bc.pinterest_image_1 <> '' AND bc.pinterest_image_2 <> '' AND bc.pinterest_image_3 <> '' AND bc.instagram_image <> '' AND bc.newsletter_featured = 1 AND bc.video = 1)";
        if ($status_filter === 'complete') {
            $where_sql .= " AND {$complete_sql}";
        } elseif ($status_filter === 'incomplete') {
            $where_sql .= " AND NOT {$complete_sql}";
        }

        if ($pinterest_filter === 'missing') {
            $where_sql .= " AND (bc.pinterest_image_1 = '' OR bc.pinterest_image_2 = '' OR bc.pinterest_image_3 = '')";
        } elseif ($pinterest_filter === 'present') {
            $where_sql .= " AND (bc.pinterest_image_1 <> '' AND bc.pinterest_image_2 <> '' AND bc.pinterest_image_3 <> '')";
        }

        if ($instagram_filter === 'missing') {
            $where_sql .= " AND bc.instagram_image = ''";
        } elseif ($instagram_filter === 'present') {
            $where_sql .= " AND bc.instagram_image <> ''";
        }

        if ($newsletter_filter === 'missing') {
            $where_sql .= " AND bc.newsletter_featured = 0";
        } elseif ($newsletter_filter === 'present') {
            $where_sql .= " AND bc.newsletter_featured = 1";
        }

        if ($video_filter === 'missing') {
            $where_sql .= " AND bc.video = 0";
        } elseif ($video_filter === 'present') {
            $where_sql .= " AND bc.video = 1";
        }

        $sql = "
            SELECT bc.*, p.post_title, p.ID as post_id
            FROM {$table} bc
            INNER JOIN {$wpdb->posts} p ON bc.post_id = p.ID
            {$where_sql}
            ORDER BY p.post_date DESC
        ";

        $rows = $args ? $wpdb->get_results($wpdb->prepare($sql, $args)) : $wpdb->get_results($sql);

        nocache_headers();
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="blogger-checklist-' . gmdate('Y-m-d') . '.csv"');

        $out = fopen('php://output', 'w');
        fputcsv($out, ['Post Title', 'Pinterest 1', 'Pinterest 2', 'Pinterest 3', 'Instagram', 'Newsletter', 'Video', 'Notes']);

        foreach ($rows as $r) {
            $title = $r->post_title ?: ('Post #' . (int)$r->post_id . ' (Deleted)');

            $safe = function(string $val): string {
                $val = (string) $val;
                if ($val !== '' && preg_match('/^[=\+\-@]/', $val)) {
                    return "'" . $val;
                }
                return $val;
            };

            fputcsv($out, [
                $safe($title),
                $safe((string)$r->pinterest_image_1),
                $safe((string)$r->pinterest_image_2),
                $safe((string)$r->pinterest_image_3),
                $safe((string)$r->instagram_image),
                ((int)$r->newsletter_featured === 1) ? 'Yes' : 'No',
                ((int)$r->video === 1) ? 'Yes' : 'No',
                $safe((string)($r->notes ?? '')),
            ]);
        }

        fclose($out);
        exit;
    }

    public function render_status_page(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', self::TEXT_DOMAIN));
        }

        global $wpdb;

        $table = $this->table_name();
        $table_exists = ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table)) === $table);
        $post_count = $table_exists ? (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table}") : 0;
        $pending_comments = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->comments} WHERE comment_approved = '0'");

        $css_exists = file_exists(plugin_dir_path(__FILE__) . 'style.css');

        $memory = ini_get('memory_limit');
        $memory_ok = ($this->parse_ini_size($memory) >= 134217728);

        $latest = get_transient('wptx_bc_latest_version');
        $latest_checked_at = get_transient('wptx_bc_latest_checked_at');

        $check_url = wp_nonce_url(admin_url('admin-post.php?action=wptx_bc_check_updates'), 'wptx_bc_check_updates');
        $update_url = wp_nonce_url(admin_url('admin-post.php?action=wptx_bc_update_now'), 'wptx_bc_update_now');

        ?>
        <div class="wrap blogger-checklist-wrap">
            <h1><?php echo esc_html__('Blogger Checklist Status', self::TEXT_DOMAIN); ?></h1>

            <div class="bc-status-actions">
                <a class="button button-primary" href="<?php echo esc_url($check_url); ?>">
                    <?php echo esc_html__('Check for Updates', self::TEXT_DOMAIN); ?>
                </a>

                <?php if (is_string($latest) && $latest !== '' && version_compare($latest, self::VERSION, '>')) : ?>
                    <a class="button button-secondary" href="<?php echo esc_url($update_url); ?>"
                       onclick="return confirm('<?php echo esc_js(sprintf(__('Install and activate version %s now?', self::TEXT_DOMAIN), $latest)); ?>');">
                        <?php echo esc_html(sprintf(__('Update Now (%s)', self::TEXT_DOMAIN), $latest)); ?>
                    </a>
                <?php endif; ?>

                <?php if (is_string($latest_checked_at) && $latest_checked_at !== '') : ?>
                    <span class="bc-muted">
                        <?php echo esc_html(sprintf(__('Last checked: %s', self::TEXT_DOMAIN), $latest_checked_at)); ?>
                    </span>
                <?php endif; ?>
            </div>

            <?php
            if (isset($_GET['bc_msg'])) {
                $msg = sanitize_text_field(wp_unslash($_GET['bc_msg']));
                if ($msg === 'checked') {
                    echo '<div class="notice notice-info"><p>' . esc_html__('Update check completed.', self::TEXT_DOMAIN) . '</p></div>';
                } elseif ($msg === 'updated') {
                    echo '<div class="notice notice-success"><p>' . esc_html__('Update installed and activated successfully.', self::TEXT_DOMAIN) . '</p></div>';
                } elseif ($msg === 'update_failed') {
                    echo '<div class="notice notice-error"><p>' . esc_html__('Update failed. Check your server logs for details.', self::TEXT_DOMAIN) . '</p></div>';
                }
            }
            ?>

            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php echo esc_html__('Check', self::TEXT_DOMAIN); ?></th>
                        <th><?php echo esc_html__('Status', self::TEXT_DOMAIN); ?></th>
                        <th><?php echo esc_html__('Details', self::TEXT_DOMAIN); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td><?php echo esc_html__('PHP Version', self::TEXT_DOMAIN); ?></td>
                        <td><?php echo version_compare(PHP_VERSION, '8.1', '>=') ? '<span class="bc-ok">✔</span>' : '<span class="bc-bad">✘</span>'; ?></td>
                        <td><?php echo esc_html('Current: ' . PHP_VERSION . '. Recommended: 8.1 or higher.'); ?></td>
                    </tr>

                    <tr>
                        <td><?php echo esc_html__('WordPress Version', self::TEXT_DOMAIN); ?></td>
                        <td><?php echo version_compare(get_bloginfo('version'), '6.0', '>=') ? '<span class="bc-ok">✔</span>' : '<span class="bc-bad">✘</span>'; ?></td>
                        <td><?php echo esc_html('Current: ' . get_bloginfo('version') . '. Recommended: 6.0 or higher.'); ?></td>
                    </tr>

                    <tr>
                        <td><?php echo esc_html__('Database Table', self::TEXT_DOMAIN); ?></td>
                        <td><?php echo $table_exists ? '<span class="bc-ok">✔</span>' : '<span class="bc-bad">✘</span>'; ?></td>
                        <td><?php echo esc_html($table . ' should exist.'); ?></td>
                    </tr>

                    <tr>
                        <td><?php echo esc_html__('Database Connection', self::TEXT_DOMAIN); ?></td>
                        <td><?php echo $wpdb->check_connection() ? '<span class="bc-ok">✔</span>' : '<span class="bc-bad">✘</span>'; ?></td>
                        <td><?php echo esc_html__('Checks if the database is accessible.', self::TEXT_DOMAIN); ?></td>
                    </tr>

                    <tr>
                        <td><?php echo esc_html__('Stylesheet', self::TEXT_DOMAIN); ?></td>
                        <td><?php echo $css_exists ? '<span class="bc-ok">✔</span>' : '<span class="bc-bad">✘</span>'; ?></td>
                        <td><?php echo esc_html('style.css should exist at ' . plugin_dir_path(__FILE__) . 'style.css'); ?></td>
                    </tr>

                    <tr>
                        <td><?php echo esc_html__('Write Permissions', self::TEXT_DOMAIN); ?></td>
                        <td><?php echo is_writable(plugin_dir_path(__FILE__)) ? '<span class="bc-ok">✔</span>' : '<span class="bc-bad">✘</span>'; ?></td>
                        <td><?php echo esc_html__('Plugin directory should be writable for updates.', self::TEXT_DOMAIN); ?></td>
                    </tr>

                    <tr>
                        <td><?php echo esc_html__('Debug Log Writable', self::TEXT_DOMAIN); ?></td>
                        <td><?php echo (is_writable(WP_CONTENT_DIR) ? '<span class="bc-ok">✔</span>' : '<span class="bc-bad">✘</span>'); ?></td>
                        <td><?php echo esc_html__('wp-content directory should be writable for logging.', self::TEXT_DOMAIN); ?></td>
                    </tr>

                    <tr>
                        <td><?php echo esc_html__('PHP Memory Limit', self::TEXT_DOMAIN); ?></td>
                        <td><?php echo $memory_ok ? '<span class="bc-ok">✔</span>' : '<span class="bc-bad">✘</span>'; ?></td>
                        <td><?php echo esc_html('Current: ' . $memory . '. Recommended: 128M or higher.'); ?></td>
                    </tr>

                    <tr>
                        <td><?php echo esc_html__('Checklist Entries', self::TEXT_DOMAIN); ?></td>
                        <td><?php echo $post_count > 0 ? '<span class="bc-ok">✔</span>' : '<span class="bc-bad">✘</span>'; ?></td>
                        <td><?php echo esc_html("Current count: {$post_count} tracked posts."); ?></td>
                    </tr>

                    <tr>
                        <td><?php echo esc_html__('Pending Comments', self::TEXT_DOMAIN); ?></td>
                        <td><?php echo ($pending_comments > 0) ? '<span class="bc-warn">⚠</span>' : '<span class="bc-ok">✔</span>'; ?></td>
                        <td><?php echo esc_html("{$pending_comments} comments awaiting moderation."); ?></td>
                    </tr>
                </tbody>
            </table>

            <p class="bc-muted">
                <?php echo esc_html__('Note: The built-in updater is a private distribution channel directly from WPTechs.NET servers.', self::TEXT_DOMAIN); ?>
            </p>
        </div>
        <?php
    }

    public function handle_check_updates(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to do that.', self::TEXT_DOMAIN));
        }
        check_admin_referer('wptx_bc_check_updates');

        $latest = $this->fetch_latest_version();
        if (is_string($latest) && $latest !== '') {
            set_transient('wptx_bc_latest_version', $latest, 6 * HOUR_IN_SECONDS);
            set_transient('wptx_bc_latest_checked_at', current_time('mysql'), 6 * HOUR_IN_SECONDS);
        }

        wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_STATUS . '&bc_msg=checked'));
        exit;
    }

    public function handle_update_now(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to do that.', self::TEXT_DOMAIN));
        }
        check_admin_referer('wptx_bc_update_now');

        $latest = get_transient('wptx_bc_latest_version');
        if (!is_string($latest) || $latest === '' || version_compare($latest, self::VERSION, '<=')) {
            wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_STATUS));
            exit;
        }

        if (!preg_match('/^\d+(\.\d+){1,3}$/', $latest)) {
            wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_STATUS . '&bc_msg=update_failed'));
            exit;
        }

        $zip = sprintf(self::UPDATE_ZIP_PATTERN, $latest);

        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/plugin.php';

        $skin = new Automatic_Upgrader_Skin();
        $upgrader = new Plugin_Upgrader($skin);

        $result = $upgrader->install($zip, ['overwrite_package' => true]);

        if (is_wp_error($result) || !$result) {
            wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_STATUS . '&bc_msg=update_failed'));
            exit;
        }

        $plugin_file = plugin_basename(__FILE__);
        $activate = activate_plugin($plugin_file);

        if (is_wp_error($activate)) {
            wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_STATUS . '&bc_msg=update_failed'));
            exit;
        }

        wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_STATUS . '&bc_msg=updated'));
        exit;
    }

    private function fetch_latest_version(): string
    {
        $resp = wp_remote_get(self::UPDATE_VERSION_URL, [
            'timeout' => 10,
            'redirection' => 3,
        ]);

        if (is_wp_error($resp)) {
            return '';
        }

        $code = (int) wp_remote_retrieve_response_code($resp);
        if ($code !== 200) {
            return '';
        }

        $body = trim((string) wp_remote_retrieve_body($resp));

        if (!preg_match('/^\d+(\.\d+){1,3}$/', $body)) {
            return '';
        }

        return $body;
    }

    public function render_tools_page(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', self::TEXT_DOMAIN));
        }

        global $wpdb;

        $dash_enabled = (int) get_option(self::OPT_DASHBOARD_CLEANER, 0);
        $bar_enabled  = (int) get_option(self::OPT_ADMINBAR_CLEANER, 0);

        $unused_tags_count = (int) $wpdb->get_var("
            SELECT COUNT(*)
            FROM {$wpdb->terms} t
            LEFT JOIN {$wpdb->term_taxonomy} tt ON t.term_id = tt.term_id
            WHERE tt.taxonomy = 'post_tag' AND tt.count = 0
        ");

        $unused_categories_count = (int) $wpdb->get_var("
            SELECT COUNT(*)
            FROM {$wpdb->terms} t
            LEFT JOIN {$wpdb->term_taxonomy} tt ON t.term_id = tt.term_id
            WHERE tt.taxonomy = 'category' AND tt.count = 0 AND t.slug != 'uncategorized'
        ");

        $save_tools_url = wp_nonce_url(admin_url('admin-post.php?action=wptx_bc_save_tools'), 'wptx_bc_save_tools');
        $del_pending_url = wp_nonce_url(admin_url('admin-post.php?action=wptx_bc_delete_pending_comments'), 'wptx_bc_delete_pending_comments');
        $del_tags_url = wp_nonce_url(admin_url('admin-post.php?action=wptx_bc_delete_unused_tags'), 'wptx_bc_delete_unused_tags');
        $del_cats_url = wp_nonce_url(admin_url('admin-post.php?action=wptx_bc_delete_unused_categories'), 'wptx_bc_delete_unused_categories');
        $empty_trash_url = wp_nonce_url(admin_url('admin-post.php?action=wptx_bc_empty_trash'), 'wptx_bc_empty_trash');
        $del_spam_url = wp_nonce_url(admin_url('admin-post.php?action=wptx_bc_delete_spam_comments'), 'wptx_bc_delete_spam_comments');
        $clear_transients_url = wp_nonce_url(admin_url('admin-post.php?action=wptx_bc_clear_expired_transients'), 'wptx_bc_clear_expired_transients');

        ?>
        <div class="wrap blogger-checklist-wrap">
            <h1><?php echo esc_html__('Neat Tools for Bloggers', self::TEXT_DOMAIN); ?></h1>

            <?php
            if (isset($_GET['bc_tools_msg'])) {
                $m = sanitize_text_field(wp_unslash($_GET['bc_tools_msg']));
                if ($m === 'saved') {
                    echo '<div class="notice notice-success"><p>' . esc_html__('Cleaner settings updated!', self::TEXT_DOMAIN) . '</p></div>';
                } elseif ($m === 'pending_deleted') {
                    echo '<div class="notice notice-success"><p>' . esc_html__('Pending comments deleted.', self::TEXT_DOMAIN) . '</p></div>';
                } elseif ($m === 'tags_deleted') {
                    echo '<div class="notice notice-success"><p>' . esc_html__('Unused tags deleted.', self::TEXT_DOMAIN) . '</p></div>';
                } elseif ($m === 'cats_deleted') {
                    echo '<div class="notice notice-success"><p>' . esc_html__('Unused categories deleted.', self::TEXT_DOMAIN) . '</p></div>';
                }
            }
            ?>

            <h2><?php echo esc_html__('Cleanup Settings', self::TEXT_DOMAIN); ?></h2>
            <form method="post" action="<?php echo esc_url($save_tools_url); ?>" class="bc-tools-form">
                <label>
                    <input type="checkbox" name="dashboard_cleaner" value="1" <?php checked($dash_enabled, 1); ?>>
                    <?php echo esc_html__('Hide all dashboard widgets except "At a Glance" and "Activity"', self::TEXT_DOMAIN); ?>
                </label>

                <label>
                    <input type="checkbox" name="admin_bar_cleaner" value="1" <?php checked($bar_enabled, 1); ?>>
                    <?php echo esc_html__('Simplify admin bar (hide New Content, Comments, and WP Logo)', self::TEXT_DOMAIN); ?>
                </label>

                <button type="submit" class="button button-primary"><?php echo esc_html__('Save Changes', self::TEXT_DOMAIN); ?></button>
            </form>

            <hr>

            <h2><?php echo esc_html__('Bulk Comment Actions', self::TEXT_DOMAIN); ?></h2>
            <p><?php echo esc_html__('Delete all pending comments from your site.', self::TEXT_DOMAIN); ?></p>
            <a class="button button-secondary" href="<?php echo esc_url($del_pending_url); ?>"
               onclick="return confirm('<?php echo esc_js(__('Are you sure you want to delete all pending comments? This cannot be undone.', self::TEXT_DOMAIN)); ?>');">
                <?php echo esc_html__('Delete All Pending Comments', self::TEXT_DOMAIN); ?>
            </a>

            <hr>

            <h2><?php echo esc_html__('Unused Tags Cleaner', self::TEXT_DOMAIN); ?></h2>
            <p><?php echo esc_html(sprintf(__('Delete all unused tags (%d found).', self::TEXT_DOMAIN), $unused_tags_count)); ?></p>
            <a class="button button-secondary" href="<?php echo esc_url($del_tags_url); ?>"
               onclick="return confirm('<?php echo esc_js(__('Are you sure you want to delete all unused tags? This cannot be undone.', self::TEXT_DOMAIN)); ?>');">
                <?php echo esc_html__('Delete Unused Tags', self::TEXT_DOMAIN); ?>
            </a>

            <hr>

            <h2><?php echo esc_html__('Unused Categories Cleaner', self::TEXT_DOMAIN); ?></h2>
            <p><?php echo esc_html(sprintf(__('Delete all unused categories except "Uncategorized" (%d found).', self::TEXT_DOMAIN), $unused_categories_count)); ?></p>
            <a class="button button-secondary" href="<?php echo esc_url($del_cats_url); ?>"
               onclick="return confirm('<?php echo esc_js(__('Are you sure you want to delete all unused categories? This cannot be undone.', self::TEXT_DOMAIN)); ?>');">
                <?php echo esc_html__('Delete Unused Categories', self::TEXT_DOMAIN); ?>
            </a>

            <hr>

            <h2>Trash Cleaner</h2>
            <p>Permanently delete items in the Trash.</p>
            <a class="button button-secondary" href="<?php echo esc_url($empty_trash_url); ?>"
            onclick="return confirm('Empty Trash now? This cannot be undone.');">Empty Trash</a>

            <hr>

            <h2>Spam Comment Actions</h2>
            <p>Delete all spam comments.</p>
            <a class="button button-secondary" href="<?php echo esc_url($del_spam_url); ?>"
            onclick="return confirm('Delete all spam comments? This cannot be undone.');">Delete Spam Comments</a>

            <hr>

            <h2>Expired Transients</h2>
            <p>Clear expired transients.</p>
            <a class="button button-secondary" href="<?php echo esc_url($clear_transients_url); ?>"
            onclick="return confirm('Clear expired transients now?');">Clear Expired Transients</a>
        </div>
        <?php
    }

    public function handle_save_tools(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to do that.', self::TEXT_DOMAIN));
        }
        check_admin_referer('wptx_bc_save_tools');

        $dash = isset($_POST['dashboard_cleaner']) ? 1 : 0;
        $bar  = isset($_POST['admin_bar_cleaner']) ? 1 : 0;

        update_option(self::OPT_DASHBOARD_CLEANER, $dash);
        update_option(self::OPT_ADMINBAR_CLEANER, $bar);

        wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_TOOLS . '&bc_tools_msg=saved'));
        exit;
    }

    public function handle_delete_pending_comments(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to do that.', self::TEXT_DOMAIN));
        }
        check_admin_referer('wptx_bc_delete_pending_comments');

        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->comments} WHERE comment_approved = '0'");

        wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_TOOLS . '&bc_tools_msg=pending_deleted'));
        exit;
    }

    public function handle_delete_unused_tags(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to do that.', self::TEXT_DOMAIN));
        }
        check_admin_referer('wptx_bc_delete_unused_tags');

        global $wpdb;
        $tags = $wpdb->get_col("
            SELECT t.term_id
            FROM {$wpdb->terms} t
            LEFT JOIN {$wpdb->term_taxonomy} tt ON t.term_id = tt.term_id
            WHERE tt.taxonomy = 'post_tag' AND tt.count = 0
        ");

        foreach ($tags as $term_id) {
            wp_delete_term((int) $term_id, 'post_tag');
        }

        wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_TOOLS . '&bc_tools_msg=tags_deleted'));
        exit;
    }

    public function handle_delete_unused_categories(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to do that.', self::TEXT_DOMAIN));
        }
        check_admin_referer('wptx_bc_delete_unused_categories');

        global $wpdb;
        $cats = $wpdb->get_col("
            SELECT t.term_id
            FROM {$wpdb->terms} t
            LEFT JOIN {$wpdb->term_taxonomy} tt ON t.term_id = tt.term_id
            WHERE tt.taxonomy = 'category' AND tt.count = 0 AND t.slug != 'uncategorized'
        ");

        foreach ($cats as $term_id) {
            wp_delete_term((int) $term_id, 'category');
        }

        wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_TOOLS . '&bc_tools_msg=cats_deleted'));
        exit;
    }

    public function maybe_clean_dashboard(): void
    {
        if (!is_admin() || !current_user_can('manage_options')) {
            return;
        }
        if (!(int) get_option(self::OPT_DASHBOARD_CLEANER, 0)) {
            return;
        }

        global $wp_meta_boxes;

        $keep = ['dashboard_glance', 'dashboard_activity'];

        if (!empty($wp_meta_boxes['dashboard'])) {
            foreach ($wp_meta_boxes['dashboard'] as $context => $priorities) {
                foreach ($priorities as $priority => $widgets) {
                    foreach (array_keys($widgets) as $id) {
                        if (!in_array($id, $keep, true)) {
                            remove_meta_box($id, 'dashboard', $context);
                        }
                    }
                }
            }
        }

        remove_meta_box('wpforms_reports_widget_lite', 'dashboard', 'normal');
        remove_meta_box('wpseo-dashboard-overview', 'dashboard', 'normal');
        remove_meta_box('woocommerce_dashboard_status', 'dashboard', 'normal');
        remove_meta_box('woocommerce_dashboard_recent_reviews', 'dashboard', 'normal');

        add_meta_box('dashboard_glance', __('At a Glance'), 'wp_dashboard_right_now', 'dashboard', 'normal', 'core');
        add_meta_box('dashboard_activity', __('Activity'), 'wp_dashboard_site_activity', 'dashboard', 'normal', 'core');
    }

    public function maybe_clean_admin_bar(\WP_Admin_Bar $wp_admin_bar): void
    {
        if (!is_admin() || !current_user_can('manage_options')) {
            return;
        }
        if (!(int) get_option(self::OPT_ADMINBAR_CLEANER, 0)) {
            return;
        }

        $wp_admin_bar->remove_node('new-content');
        $wp_admin_bar->remove_node('comments');
        $wp_admin_bar->remove_node('wp-logo');
    }

    private function parse_ini_size($val): int
    {
        $val = trim((string)$val);
        if ($val === '' || $val === '-1') {
            return PHP_INT_MAX;
        }
        $last = strtolower($val[strlen($val) - 1]);
        $num = (int) $val;
        switch ($last) {
            case 'g': $num *= 1024;
            case 'm': $num *= 1024;
            case 'k': $num *= 1024;
        }
        return $num;
    }

    public function handle_empty_trash(): void
    {
        if (!current_user_can('manage_options')) { wp_die('Nope'); }
        check_admin_referer('wptx_bc_empty_trash');

        global $wpdb;
        $ids = $wpdb->get_col("SELECT ID FROM {$wpdb->posts} WHERE post_status='trash' AND post_type IN ('post','page') LIMIT 5000");
        foreach ($ids as $id) { wp_delete_post((int)$id, true); }

        wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_TOOLS));
        exit;
    }

    public function handle_delete_spam_comments(): void
    {
        if (!current_user_can('manage_options')) { wp_die('Nope'); }
        check_admin_referer('wptx_bc_delete_spam_comments');

        global $wpdb;
        $ids = $wpdb->get_col("SELECT comment_ID FROM {$wpdb->comments} WHERE comment_approved='spam' LIMIT 10000");
        foreach ($ids as $id) { wp_delete_comment((int)$id, true); }

        wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_TOOLS));
        exit;
    }

    public function handle_clear_expired_transients(): void
    {
        if (!current_user_can('manage_options')) { wp_die('Nope'); }
        check_admin_referer('wptx_bc_clear_expired_transients');

        if (function_exists('delete_expired_transients')) { delete_expired_transients(true); }
        if (is_multisite() && function_exists('delete_expired_site_transients')) { delete_expired_site_transients(true); }

        wp_safe_redirect(admin_url('admin.php?page=' . self::MENU_SLUG_TOOLS));
        exit;
    }
}

WPTX_Blogger_Checklist::instance();
